#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "annuaire.h"

/*
 * Creation et/ou chargement de l'annuaire
 *
 * nomFichier : nom du fichier stockant l'annuaire
 *
 * renvoie l'annuaire charge ou cree
 */
Annuaire *creationChargement(char *nomFichier) {
	FILE *fich;
	Personne *p;
	Annuaire *a, *courant;
	int ret;

	/* allocation mémoire pour l'annuaire */
	a = malloc(sizeof(Annuaire));
	/* initialisation */
	a->pers = NULL;
	a->suivant = NULL;
	courant = a;

	/* ouvre le fichier en lecture */
	fich = fopen(nomFichier, "r");

	/* si le fichier existe et est lisible */
	if(fich != NULL) {
		/* tant qu'on n'a pas fini de le lire */
		while (! feof(fich)) {
			/* allocation memoire pour une nouvelle personne */
			p = malloc(sizeof(Personne));
			/* lecture de la personne */
			ret = fread(p, sizeof(Personne), 1, fich);
			/* si ça c'est bien passé */
			if(ret==1) {
				/* est-ce qu'il faut créer une nouvelle
				 cellule ? */
				if (courant->pers != NULL) {
					/* cree et remplit la cellule */
					Annuaire *cellule;
					cellule = malloc(sizeof(Annuaire));
					cellule->pers = p;
					cellule->suivant = NULL;
					/* chaînage */
					courant->suivant = cellule;
					/* avance dans le chaînage */
					courant = cellule;
				} else {
					/* remplit la cellule */
					courant->pers = p;
				}
			} else {
				/* libère la mémoire */
				free(p);
			}
		}
		/* ferme le fichier */
		fclose(fich);
	} else {
		perror("creationChargement, fopen");
	}
	
	/* renvoie l'annuaire */
	return a;
}

/*
 * Sauvegarde l'annuaire
 *
 * annu : annuaire
 * nomFichier : nom du fichier stockant l'annuaire
 *
 * renvoie 0 : si ok, -1 : si annu est vide, -2 : si erreur fichier
 */
int sauvegarde(Annuaire *annu, char *nomFichier) {
	FILE *fich;
	Annuaire *courant;
	int ret;

	/* si annu est vide */
	if(annu == NULL || annu->pers == NULL) {
		return -1;
	}

	/* ouvre le fichier en ecriture */
	fich = fopen(nomFichier, "w");

	/* si ça ne c'est pas bien passé */
	if(fich == NULL) {
		perror("sauvegarde, fopen");
		return -2;
	}
	
	/* tant qu'on a des personnes à sauver */
	courant = annu;
	while(courant != NULL) {
		if(courant->pers != NULL) {
			/* sauve la personne */
			ret = fwrite(courant->pers, sizeof(Personne), 1,
				fich);
			/* si ça ne c'est pas bien passé */
			if(ret != 1) {
				perror("sauvegarde, fwrite");
				/* tente de fermer le fichier */
				fclose(fich);
				return -2;
			}
		}
		/* passe à la personne suivante */
		courant = courant->suivant;
	}
	/* ferme le fichier */
	fclose(fich);
	return 0;
}

/*
 * Ajout d'une personne dans l'annuaire
 *
 * annu : annuaire
 * nom : nom de la personne
 * prenom : prenom de la personne
 * tel : numero de tel de la personne
 * eMail : adresse e-mail de la personne
 * adresse : adresse de la personne
 *
 * renvoie 0 : si ok, -1 : si annu est NULL, -2 : si nom est deja present
 */
int ajout(Annuaire *annu, char *nom, char *prenom, char * tel,
	  char *eMail, char *adresse) {
	Personne *p;
	Annuaire *courant;
	Annuaire *dernier;
	int ret;

	/* si annu est NULL */
	if(annu == NULL) {
		return -1;
	}

	/* verifie que la personne n'est pas présente */
	courant = annu;
	dernier = annu;
	while(courant != NULL) {
		/* s'il y a au moins une personne */
		if(courant->pers != NULL) {
			ret = strcmp(courant->pers->nom, nom);
			/* si on l'a trouve */
			if(ret == 0) {
				return -2;
			}
		}
		if(courant->suivant == NULL) {
			/* c'est le dernier */
			dernier = courant;
		}
		/* passe au suivant */
		courant = courant->suivant;
	}

	/* allocation memoire de la personne */
	p = malloc(sizeof(Personne));

	/* recopie les informations */
	strcpy(p->nom, nom);
	strcpy(p->prenom, prenom);
	strcpy(p->tel, tel);
	strcpy(p->eMail, eMail);
	strcpy(p->adresse, adresse);

	/* si l'annuaire est vide */
	if(annu->pers == NULL) {
		/* on remplit la premiere cellule */
		annu->pers = p;
	} else {
		/* crée une nouvelle cellule et la remplit */
		courant = malloc(sizeof(Annuaire));
		courant->pers = p;
		courant->suivant = NULL;
		/* ajoute la cellule à la fin de la liste */
		dernier->suivant = courant;
	}
	return 0;
}

/*
 * Effacement d'une entree de l'annuaire
 *
 * annu : annuaire
 * nom : nom de la personne
 *
 * renvoie 0 : si ok, -1 : si nom n'est pas present
 */
int effacement(Annuaire **annu, char *nom) {
	Personne *p;
	Annuaire *courant;
	Annuaire *precedant;
	Annuaire *aEffacer;
	int ret;
	int trouve = 0;
	
	/* si l'annuaire est vide */
	if(annu == NULL || *annu == NULL || (*annu)->pers == NULL) {
		/* nom ne peut pas être présent */
		return -1;
	}
	
	/* cherche la personne */
	courant = (*annu);
	precedant = NULL;
	while(!trouve && (courant != NULL)) {
		ret = strcmp(courant->pers->nom, nom);
		/* si on l'a trouve */
		if(ret == 0) {
			trouve = 1;
		} else {
			/* passe au suivant */
			precedant = courant;
			courant = courant->suivant;
		}		
	}

	if(trouve) {
		if(precedant == NULL) {
			if (courant->suivant == NULL) {
				/* il n'y a qu'une cellule dans la liste */
				/* on efface la personne */
				free(courant->pers);
				courant->pers = NULL;
			} else {
				/* il va falloir effacer la cellule*/
				aEffacer = courant;
				/* on enleve la premiere cellule */
				*annu = (*annu)->suivant;
				/* on efface la personne */
				free(aEffacer->pers);
				/* on efface la cellule */
				free(aEffacer);
			}
		} else {
			/* il va falloir effacer la cellule*/
			aEffacer = courant;
			/* on enleve la cellule de la liste */
			precedant->suivant = courant->suivant;
			/* on efface la personne */
			free(aEffacer->pers);
			/* on efface la cellule */
			free(aEffacer);
		}
		return 0;
	}
	/* il n'etait pas dans l'annuaire */
	return -1;
}

/*
 * Recherche d'une entree dans l'annuaire
 *
 * annu : annuaire
 * nom : nom de la personne
 *
 * renvoie la personne ou NULL : si nom n'est pas present
 */
Personne *recherche(Annuaire *annu, char *nom) {
	Personne *p;
	Annuaire *courant;
	int ret;
	int trouve = 0;
	
	/* si l'annuaire est vide */
	if(annu == NULL || annu->pers == NULL) {
		/* nom ne peut pas être présent */
		return NULL;
	}
	
	/* cherche la personne */
	courant = annu;
	while(!trouve && (courant != NULL)) {
		ret = strcmp(courant->pers->nom, nom);
		/* si on l'a trouve */
		if(ret == 0) {
			trouve = 1;
		} else {
			/* passe au suivant */
			courant = courant->suivant;
		}		
	}

	if(trouve) {
		/* on renvoit la personne */
		return courant->pers;
	}
	/* il n'etait pas dans l'annuaire */
	return NULL;
}

/*
 * Modification du prenom
 *
 * annu : annuaire
 * nom : nom de la personne
 * prenom : nouveau prenom
 *
 * renvoie 0 : si ok, -1 : si nom n'est pas present
 */
int modifPrenom(Annuaire *annu, char *nom, char *prenom) {
	Personne *p;
	p = recherche(annu, nom);
	if(p != NULL) {
		/* on change le prenom */
		strcpy(p->prenom, prenom);
		return 0;
	}
	/* il n'etait pas dans l'annuaire */
	return -1;
}

/*
 * Modification du tel
 *
 * annu : annuaire
 * nom : nom de la personne
 * tel : nouveau tel
 *
 * renvoie 0 : si ok, -1 : si nom n'est pas present
 */
int modifTel(Annuaire *annu, char *nom, char *tel) {
	Personne *p;
	p = recherche(annu, nom);
	if(p != NULL) {
		/* on change le tel */
		strcpy(p->tel, tel);
		return 0;
	}
	/* il n'etait pas dans l'annuaire */
	return -1;
}

/*
 * Modification de l'e-mail
 *
 * annu : annuaire
 * nom : nom de la personne
 * eMail : nouvel e-mail
 *
 * renvoie 0 : si ok, -1 : si nom n'est pas present
 */
int modifEMail(Annuaire *annu, char *nom, char *eMail) {
	Personne *p;
	p = recherche(annu, nom);
	if(p != NULL) {
		/* on change l'e-mail */
		strcpy(p->eMail, eMail);
		return 0;
	}
	/* il n'etait pas dans l'annuaire */
	return -1;
}

/*
 * Modification de l'adresse
 *
 * annu : annuaire
 * nom : nom de la personne
 * adresse : nouvelle adresse
 *
 * renvoie 0 : si ok, -1 : si nom n'est pas present
 */
int modifAdresse(Annuaire *annu, char *nom, char *adresse) {
	Personne *p;
	p = recherche(annu, nom);
	if(p != NULL) {
		/* on change l'adresse */
		strcpy(p->adresse, adresse);
		return 0;
	}
	/* il n'etait pas dans l'annuaire */
	return -1;
}

/*
 * Affiche l'annuaire
 *
 * annu : annuaire
 */
void affiche(Annuaire *annu) {
	Annuaire *courant;
	
	/* si l'annuaire est vide */
	if(annu == NULL || annu->pers == NULL) {
		printf("Annuaire vide\n");
	} else {
		/* parcourt l'annuaire */
		courant = annu;
		printf("Annuaire :\n");
		while(courant != NULL) {
			affichePersonne(courant->pers);
			courant = courant->suivant;
		}
	}
}

/*
 * Affiche une personne
 *
 * p : personne
 */
void affichePersonne(Personne *p) {
	if(p == NULL) {
		printf("Personne innexistante\n");
	} else {
		printf("Nom : %s\n", p->nom);
		printf("Prenom : %s\n", p->prenom);
		printf("Tel : %s\n", p->tel);
		printf("E-mail : %s\n", p->eMail);
		printf("Adresse : %s\n", p->adresse);
	}
}
