#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "serveur.h"
#include "annuaire.h"

/*
Protocole :

Requete = <typeReq> <parametres>\n
Reponse = <typeRep> [<parametres>]\n

typeReq = {AJOUT, EFFACE, MODIF_PRENOM, MODIF_TEL, MODIF_EMAIL, MODIF_ADRESSE, RECHERCHE}

typeRep = {OK, ERREUR, PERSONNE}

**************************************************

Reponse OK :
parametres = aucun

Reponse PERSONNE :
parametres = <nom>|<prenom>|<tel>|<email>|<adresse>

Reponse ERREUR :
parametres = <message d'erreur>

**************************************************

Requete AJOUT :
parametres = <nom>|<prenom>|<tel>|<email>|<adresse>

Exemples :
REQ : AJOUT Dupond|Jean|0123456789|dupont@gmail.com|1 rue du Taur 31000 Toulouse\n
REP : OK\n

REQ : AJOUT Dupond|Jean|0123456789|dupont@gmail.com|1 rue du Taur 31000 Toulouse\n
REP : ERREUR la personne existe deja\n

****************

Requete EFFACE :
parametres = <nom>

Reponses possibles : OK ou ERREUR

Exemples :
REQ : EFFACE Dupond\n
REP : OK\n

REQ : EFFACE Dupond\n
REP : ERREUR la personne n'existe pas\n

****************

Requete MODIF_PRENOM :
parametres = <nom>|<prenom>

Reponses possibles : OK ou ERREUR

Exemples :
REQ : MODIF_PRENOM Dupond|Jacques\n
REP : OK\n

REQ : MODIF_PRENOM Dupond|Jacques\n
REP : ERREUR la personne n'existe pas\n

****************

Idem MODIF_TEL, MODIF_EMAIL, MODIF_ADRESSE

****************

Requete RECHERCHE :
parametres = <nom>

Reponses possibles : PERSONNE ou ERREUR

Exemples :

REQ : RECHERCHE Dupond\n
REP : PERSONNE Dupond|Jacques|0123456789|dupont@gmail.com|1 rue du Taur 31000 Toulouse\n

REQ : RECHERCHE Dupond\n
REP : ERREUR la personne n'existe pas\n

*/

/* 
 * Types de requêtes
 */

#define REQ_INCONNUE	0
#define AJOUT 		1
#define EFFACE 		2
#define MODIF_PRENOM 	3
#define MODIF_TEL 	4
#define MODIF_EMAIL 	5
#define MODIF_ADRESSE 	6
#define RECHERCHE	7

/*
 * Execute une requete ajout
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeAjout(Annuaire *a, char *parametres) {
	char nom[256];
	char prenom[256];
	char tel[32];
	char eMail[256];
	char adresse[1024];
	int ret;

	/* decoupe les parametres */
	ret = sscanf(parametres, "%[^|]|%[^|]|%[^|]|%[^|]|%[^|]", 
			nom, prenom, tel, eMail, adresse);

	if(ret != 5) {
		return "ERREUR Requête mal formée\n";
	} else {
		/* ajoute à l'annuaire */
		ret = ajout(a, nom, prenom, tel, eMail, adresse);
		/* prépare la réponse */
		if(ret == 0) {
			/* l'annuaire a été modifié, on le sauve */
			ret = sauvegarde(a, "annu.bin");
			if(ret < 0) printf("erreur sauvegarde : %d\n", ret);
			return "OK\n";
		} else if(ret == -2) {
			return "ERREUR la personne existe deja\n";
		} else {
			return "ERREUR interne du serveur\n";
		}
	}
}

/*
 * Execute une requete efface
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeEfface(Annuaire **a, char *parametres) {
	if(parametres == NULL) {
		return "ERREUR Requête mal formée\n";
	} else {
		int ret;
		/* efface de l'annuaire */
		ret = effacement(a, parametres);
		/* prépare la réponse */
		if(ret == 0) {
			/* l'annuaire a été modifié, on le sauve */
			ret = sauvegarde(*a, "annu.bin");
			if(ret < 0) printf("erreur sauvegarde : %d\n", ret);
			return "OK\n";
		} else if(ret == -1) {
			return "ERREUR la personne n'existe pas\n";
		} else {
			return "ERREUR interne du serveur\n";
		}
	}
}

/*
 * Execute une requete modif prenom
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeModifPrenom(Annuaire *a, char *parametres) {
	char nom[256];
	char modif[1024];
	int ret;

	/* decoupe les parametres */
	ret = sscanf(parametres, "%[^|]|%[^|]", nom, modif);

	if(ret != 2) {
		return "ERREUR Requête mal formée\n";
	} else {
		/* modifie l'annuaire */
		ret = modifPrenom(a, nom, modif);
		/* prépare la réponse */
		if(ret == 0) {
			/* l'annuaire a été modifié, on le sauve */
			ret = sauvegarde(a, "annu.bin");
			if(ret < 0) printf("erreur sauvegarde : %d\n", ret);
			return "OK\n";
		} else if(ret == -1) {
			return "ERREUR la personne n'existe pas\n";
		} else {
			return "ERREUR interne du serveur\n";
		}
	}
}

/*
 * Execute une requete modif tel
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeModifTel(Annuaire *a, char *parametres) {
	char nom[256];
	char modif[1024];
	int ret;

	/* decoupe les parametres */
	ret = sscanf(parametres, "%[^|]|%[^|]", nom, modif);

	if(ret != 2) {
		return "ERREUR Requête mal formée\n";
	} else {
		/* modifie l'annuaire */
		ret = modifTel(a, nom, modif);
		/* prépare la réponse */
		if(ret == 0) {
			/* l'annuaire a été modifié, on le sauve */
			ret = sauvegarde(a, "annu.bin");
			if(ret < 0) printf("erreur sauvegarde : %d\n", ret);
			return "OK\n";
		} else if(ret == -1) {
			return "ERREUR la personne n'existe pas\n";
		} else {
			return "ERREUR interne du serveur\n";
		}
	}
}

/*
 * Execute une requete modif e-mail
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeModifEMail(Annuaire *a, char *parametres) {
	char nom[256];
	char modif[1024];
	int ret;

	/* decoupe les parametres */
	ret = sscanf(parametres, "%[^|]|%[^|]", nom, modif);

	if(ret != 2) {
		return "ERREUR Requête mal formée\n";
	} else {
		/* modifie l'annuaire */
		ret = modifEMail(a, nom, modif);
		/* prépare la réponse */
		if(ret == 0) {
			/* l'annuaire a été modifié, on le sauve */
			ret = sauvegarde(a, "annu.bin");
			if(ret < 0) printf("erreur sauvegarde : %d\n", ret);
			return "OK\n";
		} else if(ret == -1) {
			return "ERREUR la personne n'existe pas\n";
		} else {
			return "ERREUR interne du serveur\n";
		}
	}
}

/*
 * Execute une requete modif adresse
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeModifAdresse(Annuaire *a, char *parametres) {
	char nom[256];
	char modif[1024];
	int ret;

	/* decoupe les parametres */
	ret = sscanf(parametres, "%[^|]|%[^|]", nom, modif);

	if(ret != 2) {
		return "ERREUR Requête mal formée\n";
	} else {
		/* modifie l'annuaire */
		ret = modifAdresse(a, nom, modif);
		/* prépare la réponse */
		if(ret == 0) {
			/* l'annuaire a été modifié, on le sauve */
			ret = sauvegarde(a, "annu.bin");
			if(ret < 0) printf("erreur sauvegarde : %d\n", ret);
			return "OK\n";
		} else if(ret == -1) {
			return "ERREUR la personne n'existe pas\n";
		} else {
			return "ERREUR interne du serveur\n";
		}
	}
}

/*
 * Execute une requete recherche
 *
 * a : annuaire
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeRecherche(Annuaire *a, char *parametres) {
	if(parametres == NULL) {
		return "ERREUR Requête mal formée\n";
	} else {
		Personne *p;
		/* recherche dans l'annuaire */
		p = recherche(a, parametres);
		/* prépare la réponse */
		if(p != NULL) {
			char reponse[4096];
			sprintf(reponse, "PERSONNNE %s|%s|%s|%s|%s\n", p->nom, p->prenom,
				p->tel, p->eMail, p->adresse);
			/* NB : il faudra faire un free */
			return strdup(reponse);
		} else {
			return strdup("ERREUR la personne n'existe pas\n");
		}
	}
}

/*
 * Execute une requete
 *
 * a : annuaire
 * type : type de requete
 * parametres : parametres de la requete
 *
 * renvoit : la reponse
 */
char *executeRequete(Annuaire **a, int type, char *parametres) {
	char *reponse;
	/* AJOUT, EFFACE, MODIF_PRENOM, MODIF_TEL, MODIF_EMAIL, MODIF_ADRESSE, RECHERCHE */
	switch(type) {
		case AJOUT:
			reponse = executeAjout(*a, parametres);
			break;
		case EFFACE:
			reponse = executeEfface(a, parametres);
			break;
		case MODIF_PRENOM:
			reponse = executeModifPrenom(*a, parametres);
			break;
		case MODIF_TEL:
			reponse = executeModifTel(*a, parametres);
			break;
		case MODIF_EMAIL:
			reponse = executeModifEMail(*a, parametres);
			break;
		case MODIF_ADRESSE:
			reponse = executeModifAdresse(*a, parametres);
			break;
		case RECHERCHE:
			reponse = executeRecherche(*a, parametres);
			break;
		default:
			reponse = "ERREUR requete inconnue\n";
			break;
	}
	return reponse;
}

/*
 * Decoupe et execute une requete
 *
 * a : annuaire
 * requete : requete
 */
void decoupeEtExecute(Annuaire **a, char *requete, Client *cl) {
	char *reponse;
	int type;
	char *parametres;
	int lgParam;

	/* compare le debut de la requête avec les types connus */
	if(strncmp(requete, "AJOUT", 5)==0) {
		type = AJOUT;
	} else if(strncmp(requete, "EFFACE", 6)==0) {
		type = EFFACE;
	} else if(strncmp(requete, "MODIF_PRENOM", 12)==0) {
		type = MODIF_PRENOM;
	} else if(strncmp(requete, "MODIF_TEL", 9)==0) {
		type = MODIF_TEL;
	} else if(strncmp(requete, "MODIF_EMAIL", 11)==0) {
		type = MODIF_EMAIL;
	} else if(strncmp(requete, "MODIF_ADRESSE", 13)==0) {
		type = MODIF_ADRESSE;
	} else if(strncmp(requete, "RECHERCHE", 9)==0) {
		type = RECHERCHE;
	} else {
		/* type inconnu */
		type = REQ_INCONNUE;
	}
	/* recupere les paramètres */
	parametres = strstr(requete, " ");
	if(parametres != NULL) {
		/* saute l'espace */
		parametres++;
		/* enlève \r et \n à la fin de parametres */
		lgParam = strlen(parametres);
		if(parametres[lgParam-2] == '\r') {
			parametres[lgParam-2] = '\0';
		}
		if(parametres[lgParam-1] == '\n') {
			parametres[lgParam-1] = '\0';
		}
	}
	/* execute la requête */
	reponse = executeRequete(a, type, parametres);
	/* envoie la réponse */
	printf("Ma réponse : %s\n", reponse);
	Emission(reponse, cl);
	/* libère la mémoire si nécessaire */
	if(type == RECHERCHE) {
		free(reponse);
	}
}

int main() {
	Annuaire *a;
	char *message = NULL;
	Client *cl;

	Initialisation();
	a = creationChargement("annu.bin");
	affiche(a);
	while(1) {
		cl = AttenteClient();
		do {
			if(message != NULL) {
				free(message);
			}
	
			message = Reception(cl);
	
			if(message != NULL) {
				printf("J'ai recu: %s\n", message);
	
				/* decoupage et execution de la requete */
				decoupeEtExecute(&a, message, cl);
			}
		} while(message != NULL);
	
		TerminaisonClient(cl);
	}
	return 0;
}
